#!/bin/sh
#
# Copyright 2016, 2019 VMware, Inc. All rights reserved.
#
# This script is used to change the hostname in a deployed SVA instance
#
# This includes :
#     - Updating the OS ( depending on flags )
#     - Update the certificate, if using a self-signed cert
#     - Update configuration files
#     - Update database tables
#
# syntax: changeHostName.sh <new hostname> [<old hostname>]
#
# Environment Variables:
#   # vra_db_pwd=`cat /usr/local/horizon/DatabaseConfig/db.pwd`
    #  export PGPASSWORD=$vra_db_pwd : password used to update the database
#
. /usr/local/horizon/scripts/flag.inc
. /usr/local/horizon/scripts/sslCommons.hzn

export PATH=$PATH:/opt/vmware/vpostgres/9.6/bin/

if [ ! -f $FLAG_IDM_INSTALLED ]; then
  echo Host name change only supported for IDM installations
  exit 99
fi

NEW_HOST=$1
if [ -z "$NEW_HOST" ]; then
  echo New host not specified
  exit 1
fi

#Determine the current hostname
CURRENT_HOST=$(hostname -f)
OLD_HOST=${2:-${CURRENT_HOST}}

LOG_FILE=/usr/local/horizon/log/changeHostName.log
echo > ${LOG_FILE}
say() {
  echo `date '+%H:%M:%S'`" :: $*" | tee -a ${LOG_FILE}
}

if [ -z "$JAVA_HOME" ] ; then
  JAVA_HOME="/usr/java/jre-vmware"
fi

RUNTIME_CONFIG=/usr/local/horizon/conf/runtime-config.properties
. /usr/local/horizon/scripts/manageTcCfg.inc
. /usr/local/horizon/scripts/utils.inc

########## Verify DB access
JDBC_URL=$(getPropertyVal datastore.jdbc.url ${RUNTIME_CONFIG})
JDBC_USER=$(getPropertyVal datastore.jdbc.userName ${RUNTIME_CONFIG})

IFS='/' read -a JDBC_URL_PARTS <<< "${JDBC_URL}"
JDBC_DB=${JDBC_URL_PARTS[3]%%\?*}

IFS=':' read -a SCHEME_PARTS  <<< "${JDBC_URL_PARTS[0]}"
if [ "${SCHEME_PARTS[1]}" != "postgresql" ]; then
  say "Database update only supported for Postgres databases"
  exit 3
fi

IFS=':' read -a SERVER_PARTS  <<< "${JDBC_URL_PARTS[2]}"
JDBC_HOST=${SERVER_PARTS[0]}
JDBC_PORT=${SERVER_PARTS[1]:-5432}
if [ -z "$JDBC_HOST" ]; then
  say Unable to extract host from JDBC URL
  exit 4
fi


if [[ $JDBC_SCHEMA =~ .*currentSchema=.* ]]; then
  URL_SCHEMA=$(echo ${JDBC_URL_PARTS[3]} | sed -re 's/^.*currentSchema=([^&]+).*$/\1/')
fi

JDBC_SCHEMA=${URL_SCHEMA:-saas}

# If PGPASSWORD is set, keep it.  Otherwise, use the default password
# if [ -z "$PGPASSWORD" ]; then
#   if test -e /usr/local/horizon/scripts/manageDb.inc ;then
#      . /usr/local/horizon/scripts/manageDb.inc
#     export PGPASSWORD=$(getDefaultDbPwd)
#    fi
#fi
#if [ -z "$PGPASSWORD" ]; then
#    say Unable to determine the database password, please set PGPASSWORD and run the script again.
#    exit 5
#fi

say Verifying database connectivity

echo Database connection info: ${JDBC_USER}:${JDBC_HOST}:${JDBC_PORT}:${JDBC_DB}:${JDBC_SCHEMA} >> ${LOG_FILE}

# export the vra Database password for this change hostname session
vra_db_pwd=`cat /usr/local/horizon/DatabaseConfig/db.pwd`
export PGPASSWORD=$vra_db_pwd

PSQL_CONN_PARAMS="-h ${JDBC_HOST} -p ${JDBC_PORT} -U ${JDBC_USER} ${JDBC_DB}"
psql ${PSQL_CONN_PARAMS} -c "select count(id) from \"${JDBC_SCHEMA}\".\"GlobalConfigParameters\";" > /dev/null
if [ $? -ne 0 ]; then
    say Database connection check failed for ${JDBC_USER} on ${JDBC_HOST}:${JDBC_PORT} with ${JDBC_DB}.${JDBC_SCHEMA}
    exit 6
fi

########## begin work to change the hostname
say Starting hostname change, full output is available in $LOG_FILE

isRunning=`service horizon-workspace status | grep -c "RUNNING as"`
if [ "$isRunning" -gt 0 ]; then
  say Stopping web server...
  service horizon-workspace stop >> ${LOG_FILE} 2>&1
fi

########## Update certificate only if we are using the self-signed cert

KEYTOOL=${JAVA_HOME}/bin/keytool

cert=`${KEYTOOL} -list -rfc -alias $(getInstanceSSLAlias) -storepass $(getInstanceKeystorePwd) -keystore ${INSTANCE_KEYSTORE} -provider org.bouncycastle.jcajce.provider.BouncyCastleFipsProvider -providerpath /usr/local/horizon/jre-endorsed/bc-fips-1.0.1.BC-FIPS-Certified.jar -storetype BCFKS | openssl x509 -out CERT`

say cert CERT

testCertificateCompliance CERT "$NEW_HOST"
if [ $? -ne 30 ]
then
  say testCertificateCompliance CERT "$NEW_HOST"
  say Certificate did not match the new host - regenerating a self signed certificate
  HOST=`echo $NEW_HOST | cut -f 2,3,4,5,6  -d '.'`
  /usr/local/horizon/scripts/secure/wizardssl.hzn --makesslcert $NEW_HOST $NEW_HOST *.$HOST,$NEW_HOST,vsphere.local.$HOST >> ${LOG_FILE} 2>&1
  /usr/local/horizon/scripts/secure/wizardssl.hzn >> ${LOG_FILE} 2>&1
fi

function replaceInTable(){
  local TABLE=$1
  local ID_FIELD=$2
  local VAL_FIELD=$3
  local FIELD_SEP='|'

  say Processing: $TABLE
  local RESULTS=("$(psql -q -F${FIELD_SEP} -A -t ${PSQL_CONN_PARAMS} -c "select \"${ID_FIELD}\",\"${VAL_FIELD}\" from \"${JDBC_SCHEMA}\".\"${TABLE}\" where \"${VAL_FIELD}\" like '%${OLD_HOST}%';")")

  for r in "${RESULTS[@]}"; do
    local ID=${r%%${FIELD_SEP}*}
    local REC=${r##*${FIELD_SEP}}
    local NREC=${REC//${OLD_HOST}/${NEW_HOST}}
    psql ${PSQL_CONN_PARAMS} -c "UPDATE \"${JDBC_SCHEMA}\".\"${TABLE}\" set \"${VAL_FIELD}\"='${NREC}' WHERE \"${ID_FIELD}\"=${ID};" >> ${LOG_FILE} 2>&1
  done
}

function deleteInTable(){
  local TABLE=$1
  local ID=$2
  local HOST=$3

  say Processing: $TABLE
    psql ${PSQL_CONN_PARAMS} -c "DELETE FROM \"${JDBC_SCHEMA}\".\"${TABLE}\" WHERE \"${ID}\"='${HOST}';" >> ${LOG_FILE} 2>&1
}

function updateInTable(){

  psql ${PSQL_CONN_PARAMS} -c "update \"${JDBC_SCHEMA}\".\"Connector\" set \"host\"= '${NEW_HOST}' where \"host\" = '${OLD_HOST}';" >> ${LOG_FILE} 2>&1
}

function updateIdentityProviders()
{

COUNT=`psql $PSQL_CONN_PARAMS -q -A -t -c "select COUNT(*)  from \"${JDBC_SCHEMA}\".\"FederationArtifacts\" where \"strData\" like '%${OLD_HOST}%';"`

while [ $COUNT -ne 0 ]
do
  psql $PSQL_CONN_PARAMS -q -A -t -c "select \"strData\" from \"${JDBC_SCHEMA}\".\"FederationArtifacts\" where \"strData\" like '%${OLD_HOST}%' LIMIT 1;" > blah
  ID=`psql $PSQL_CONN_PARAMS -q -A -t -c "select \"id\" from \"${JDBC_SCHEMA}\".\"FederationArtifacts\" where \"strData\" like '%${OLD_HOST}%' LIMIT 1;"`
  sed -e "s/$OLD_HOST/$NEW_HOST/g" blah > blah2
  GW=`cat blah2`
  rm blah blah2
  psql $PSQL_CONN_PARAMS -q -A -t -c "update \"${JDBC_SCHEMA}\".\"FederationArtifacts\" set \"strData\"='${GW}' where \"id\"=$ID;"
  COUNT=$(( $COUNT - 1 ))
done

}

function disable_kerb_securid()
{
TMP=/tmp/tmp$$
for cfg in `find /usr/local/horizon/conf -name config-state.json` ; do

cat $cfg | /usr/bin/python -c '
# use python because it has a very nice json parser
import sys, json;

# load the json into myobj
myobj = json.load(sys.stdin);

myobj["enabled"] = False;

# disable the Kerberos Adapter if it is there
try:
    if myobj["idpAdapterConfig"]["com.vmware.horizon.adapters.kerberosAdapter.KerberosIdpAdapter"]["enabled"] :
        myobj["idpAdapterConfig"]["com.vmware.horizon.adapters.kerberosAdapter.KerberosIdpAdapter"]["enabled"] = "false";
except:
    pass

# disable the SecureID Adapter if it is there
try:
    if myobj["idpAdapterConfig"]["com.vmware.horizon.adapters.securAdapter.SecurIDIdpAdapter"]["enabled"] :
        myobj["idpAdapterConfig"]["com.vmware.horizon.adapters.securAdapter.SecurIDIdpAdapter"]["enabled"] = "false";
except:
    pass

# output prettified json similar to what the java code does
print json.dumps(myobj, sort_keys=True, indent=2, separators=(",", " : "));
' > $TMP
        if [ -s $TMP ]; then
            cp $TMP $cfg
        fi
        rm -f $TMP
    done
    rm -f /opt/vmware/horizon/securid
}

########## Update the database
say Updating database tables
replaceInTable OAuth2Client id redirectUri
replaceInTable Connector id host
updateInTable
#replaceInTable FederationArtifacts id strData
#replaceInTable ServiceInstance id hostName
deleteInTable ServiceInstance hostName $OLD_HOST
#disable_kerb_securid
updateIdentityProviders


# Restart the web server, if needed
isRunning=`service horizon-workspace status | grep -c "RUNNING as"`
if [ "$isRunning" -eq 0 ]; then
  say Starting web server...
  service horizon-workspace start >> ${LOG_FILE} 2>&1
fi
sh /usr/local/horizon/scripts/login-ad-user.sh

say hostname update complete
