#!/usr/bin/perl
#
# Copyright 2008-2014 VMware, Inc.  All rights reserved.
#
#

use strict;

use File::Copy;
use File::Basename;
use Getopt::Long;
use lib '/usr/local/horizon/scripts';
use Properties;

# Property file where the URLs are maintained
my $WEBAPPS_ROOT="/opt/vmware/horizon/workspace/webapps";
my $PROP="/usr/local/horizon/conf/client-download.properties";
my $CLIENTFOLDER="client";
my $PUBLICFOLDER= "$WEBAPPS_ROOT/ROOT/$CLIENTFOLDER";
my $CDSFOLDER= "$WEBAPPS_ROOT/ROOT/$CLIENTFOLDER/cds";

# Download location, this is actually accessed as https://<gw>/download
my $OCT="$WEBAPPS_ROOT/ROOT/$CLIENTFOLDER";

my $USER="horizon";
my $GROUP="www";
my $xmlVersion;
my $xmlFullVersion;
my $tarballAbsoluteURL;
my $file;
my $zero_ver = "0.0.0-0000";

my $install;
my $update;
my @new_ver_val;
my @old_ver_val;

# Now the version is extracted out of the zip file. So we need to make this global.
my $new_ver;
my $result = GetOptions ("install" => \$install,
	"update" => \$update,
	"clientfile=s" => \$file);

if (defined $file)
{
	# Convert the file name to an absolute name
	$file = File::Spec->rel2abs( $file );
	unless (-e $file)
	{
		die "Missing the clientfile location: $file \n";
	}
	system ("file $file | grep -q \"Zip archive data\"") and die "not a zip file";
}

if (defined $install)
{
	# Check if there is a new install and download.
	compare_version($tarballAbsoluteURL);
	download_clientsZip();
	cleanup();
}
elsif (defined $update)
{
	# Check if there is a new update
	compare_version($tarballAbsoluteURL);
	cleanup();
}
else
{
	usage();
}

sub usage()
{
	print "Need to specify either --install or --update.\n";
	print "If --install is specified, then it will download and install the client updates.\n";
	print "If the file is available locally use --clientfile <zipfile> to check for update or install the update.\n";
	exit 0;
}

sub cleanup(){
	system("rm -rf /tmp/updates$$");
}

sub compare_hash
{
	if ($new_ver_val[0] > $old_ver_val[0])
	{
		return 1;
	}
	if ($new_ver_val[0] < $old_ver_val[0])
	{
		return 0;
	}
	if ($new_ver_val[1] > $old_ver_val[1])
	{
		return 1;
	}
	if ($new_ver_val[1] < $old_ver_val[1])
	{
		return 0;
	}
	if ($new_ver_val[2] > $old_ver_val[2])
	{
		return 1;
	}
	if ($new_ver_val[2] < $old_ver_val[2])
	{
		return 0;
	}
	if ($new_ver_val[3] > $old_ver_val[3])
	{
		return 1;
	}
	return 0;
}
sub compare_version()
{
	my $url = shift;
	system ("mkdir -p /tmp/updates$$");
	chdir ("/tmp/updates$$") or die "Unable to cd to /tmp/updates$$\n";
	if (defined $url)
	{
		# url is only defined if we go through the online manifest.xml
		getstore($url, $file) or die "Error on downloading $url";
	}
	else
	{
		# file was provided using --clientfile
		system("unzip -t $file >/dev/null 2>&1 || exit");
	}
	system("unzip $file > /dev/null");
	($new_ver) = `cat version.txt`;
	chomp($new_ver);
	($new_ver)  =~ s/\-/\./g;
	($new_ver) =~ m/(\d+.\d+.\d+.\d*)/g;
	@new_ver_val = split(/\./, $new_ver);
	my $old_ver;
	# Properties file must be present to update the windows/mac version
	if (-e $PROP)
	{
		# Check the old version against the clients-<version>.zip in ZoConstants.properties file
		$old_ver = `basename \`ls -t1 $PUBLICFOLDER/clients*.zip 2>/dev/null | head -1\` 2>/dev/null`;
		chomp($old_ver);
	        if ($old_ver eq "" )
		{
			# Make it version 0.00
	                $old_ver=$zero_ver;
       		 }
	}
	else
	{
		die "missing $PROP file\n";
	}
	($old_ver)  =~ s/\-/\./g;
	($old_ver) = $old_ver =~ m/(\d+.\d+.\d+.\d*)/g;
	@old_ver_val = split(/\./, $old_ver);
	if  (compare_hash())
	{
		if (defined $update)
		{
			print "Newer update available: $file\n";
			print "New version: $new_ver\n";
			if ("$old_ver" == "$zero_ver")
			{
				print "Old version: No zip present\n";
			}
			else
			{
				print "Old version: $old_ver\n";
			}
		}
		else
		{
			print "New version: $new_ver\n";
			if ("$old_ver" == "$zero_ver")
			{
				print "Old version: No zip present\n";
			}
			else
			{
				print "Old version: $old_ver\n";
			}
			print "Applying update $file\n";
		}
	}
	else
	{
		print "No new updates available\n";
		exit 0;
	}
}

sub download_clientsZip($)
{
	# Remove the existing zip files as the new one is copied
	system("rm -f $OCT/*.zip");
	system("mkdir -p $CDSFOLDER");
	system("cp $file $OCT");

	chdir ("/tmp/updates$$") or die "Unable to cd to /tmp/updates$$\n";
	# Assumption is that the tar ball will contain
	# *.dmg file --> Mac Client
	# *.exe file -> Windows Client
	# Please update accordingly.
	print "Moving the contents of cds folder to $CDSFOLDER...\n";
	system("mv $CDSFOLDER $CDSFOLDER$$");
	system("mkdir -p $CDSFOLDER");
	system("cp -rf cds/* $CDSFOLDER");
	system("chmod -R a+rX $CDSFOLDER");
	system("chown -R $USER:$GROUP $PUBLICFOLDER");

	# Remove the downloaded file
	system("rm -f $file");

	my $OLDEXE = `basename $OCT/*.exe`;
	my $old_win_version = $OLDEXE;
	( $old_win_version ) = $old_win_version =~ /VMware-.*-Desktop-([0-9-\.]+)\.exe$/;
	my $EXE = `ls *.exe`;
	chomp($EXE);
	my $new_win_version = $EXE;
	my $prop = "/$CLIENTFOLDER/$EXE";
	( $new_win_version ) = $new_win_version =~ /VMware-.*-Desktop-([0-9-\.]+)\.exe$/;
	($old_win_version)  =~ s/\-/\./g;
	($new_win_version)  =~ s/\-/\./g;
	@new_ver_val = split(/\./, $new_win_version);
	@old_ver_val = split(/\./, $old_win_version);
	print "Moving the exe file to $OCT\n";
	system("mv *.exe $OCT");
	if (compare_hash())
	{
		system("chown $USER:$GROUP $OCT/*.exe");
		system("chmod 644 $OCT/*.exe");
		# This will update the download URL for win
		print "Updating the $PROP with Windows client version\n";
		changeProperty("winSyncClientUrl",($prop));
	}
	else
	{
		print "A newer version of exe $old_win_version is already present in the $PROP file.\n";
	}

	# /tmp/updates$$ gets removed, we need to change dir
	chdir("/");
	system ("rm -rf /tmp/updates$$");

	print "System updated successfully\n";
	my $dirs = `ls $PUBLICFOLDER/ | grep  '^cds' | grep -v 'cds\$'`;
	print "NOTE: Administrator is responsible to remove the old cds\n";
	print "$dirs directories in $PUBLICFOLDER folder\n";
}


sub getProperty($)
{
        open PROPS, "< $PROP"
                or die "unable to open $PROP file";

        my $properties = new Properties();
        $properties->load(*PROPS);
        my $value = $properties->getProperty(shift);
        close PROPS;
        return $value;
}

sub changeProperty()
{
        open PROPS, "< $PROP"
                or die "unable to open $PROP file";

        my $properties = new Properties(wrap=>0);
        $properties->load(*PROPS);
        open PROPS, "> $PROP"
                or die "unable to open $PROP file";
        $properties->changeProperty(shift,shift);
	$properties->format( '%s = %s' );
        $properties->store (*PROPS);
        close PROPS;
}
