#!/bin/sh

#
#  This script is responsible for installing Tomcat Server and generating the init.d script
#
#  ./installTcServer.sh [-u tomcat_user -n instance_name -b instance_basedir -i init_script_name -o optional_packages -I <no init script> -T <no tc server> -U <upgrade instance> ]

usage () {
      echo "$0 [-u tomcat_user -g tomcat_group -p port -R -n instance_name -b instance_basedir -i init_script_name -o optional_packages -I -T -U ]"
}

. /usr/local/horizon/scripts/distroSpecific.inc
. /usr/local/horizon/scripts/flag.inc

TOMCAT_USER=nobody
TOMCAT_GROUP=www
INSTANCE_NAME=
INSTANCE_BASE=/opt/vmware/horizon
INSTANCE_INIT_NAME=horizon
REQUIRED=
OPTIONAL=
PORT=443
HTTP_PORT=80
UPGRADE=0
HTTPS_PASSTHROUGH_PORT=7443

INIT_SCRIPT=1
DEPLOY_TCSERVER=1

TOMCAT_SERVER_HOME=/usr/share/tomcat
TOMCAT_MIGRATION=0

while getopts "u:g:n:p:b:i:r:o:h:ITU" OPTION
do
     case $OPTION in
         u) TOMCAT_USER=$OPTARG;;
         g) TOMCAT_GROUP=$OPTARG;;
         n) INSTANCE_NAME=$OPTARG;;
         b) INSTANCE_BASE=$OPTARG;;
         p) PORT=$OPTARG;;
         h) HTTP_PORT=$OPTARG;;
         i) INSTANCE_INIT_NAME=$OPTARG;;
         r) REQUIRED=$OPTARG;;
         o) OPTIONAL="$OPTARG";;
         I) INIT_SCRIPT=0;;
         T) DEPLOY_TCSERVER=0;;
         U) UPGRADE=1;;
         ?)
                usage;
                exit 1;;
     esac
done

if test -z "${JAVA_HOME}" ; then
  export JAVA_HOME=/usr/java/jre-vmware
fi

getNewTemplate(){
  if [[ $TOMCAT_MAJOR_VERSION -lt 8 || $TOMCAT_MAJOR_VERSION -eq 8 && $TOMCAT_MINOR_VERSION -lt 5 ]]; then
      echo bio-ssl
  else
      echo nio-ssl
  fi
}

if test ${DEPLOY_TCSERVER} -gt 0 ; then
  #disable the Java extension directory, as it may have an incompatible version of bc installed
  export JAVA_OPTS="-Djava.ext.dirs=/dev/null"

  DEFAULT_PWD_ARGS=
  TEMPLATE=nio-ssl

  ${TOMCAT_SERVER_HOME}/instance/bin/tomcat-instance-create.sh --instance-location $INSTANCE_BASE --instance-name $INSTANCE_NAME \
          --instance-user $TOMCAT_USER \
          --provides $INSTANCE_INIT_NAME \
          --should-start "$OPTIONAL" \
          --should-stop "$OPTIONAL" \
          --description 'IDM tomcat server instance' \
          --chkconfig '35 70 30'

  if test $? -ne 0 ; then
    echo Failed to install tomcat server
    exit 1
  fi

  COPY_CONF_FILES=( conf/catalina.policy
                    conf/catalina.properties
                    conf/context.xml
                    conf/web.xml )
  DEST_FOLDER="${INSTANCE_BASE}/${INSTANCE_NAME}"

  echo Copying configuration files

  for f in ${COPY_CONF_FILES[@]} ; do
    TO_COPY=${TOMCAT_SERVER_HOME}/${f}
    if test -e $TO_COPY ; then
      cp -v -R -f ${TO_COPY} ${DEST_FOLDER}/${f}
    fi
  done

  if [ -f $FLAG_FIPS_MODE ] ; then
    KEYSTORE_TYPE=BCFKS
    KEYSTORE_PROVIDER=BCFIPS
  else
    KEYSTORE_TYPE=JKS
    KEYSTORE_PROVIDER=SUN
  fi

  if [ ! -f "$FLAG_TOMCAT_MIGRATION" ] ; then
    echo "base.shutdown.port=-1" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "base.jmx.port=6969" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "nio-ssl.ssl.key.alias=tcserver" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "ssl.passthrough.alias=passthrough" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "nio-ssl.ssl.keystore.location=conf/tcserver.keystore" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "nio-ssl.ssl.keystore.type=${KEYSTORE_TYPE}" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "nio-ssl.https.port=${PORT}" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "http.port=${HTTP_PORT}" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "https.passthrough.port=${HTTPS_PASSTHROUGH_PORT}" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "bind.address=0.0.0.0" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    echo "nio-ssl.ssl.keystore.provider=${KEYSTORE_PROVIDER}" >> ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
  fi

  chown -R ${TOMCAT_USER}:${TOMCAT_GROUP} ${INSTANCE_BASE}/${INSTANCE_NAME}

else
  echo Not installing tomcat server
fi

if test ${INIT_SCRIPT} -gt 0 ; then
  echo Generating init script /etc/init.d/${INSTANCE_INIT_NAME}
  cp -v -f ${INSTANCE_BASE}/${INSTANCE_NAME}/bin/init.d.sh /etc/init.d/${INSTANCE_INIT_NAME}

  if test $? -ne 0 ; then
    echo Failed to install tomcat server init script
    exit 1
  fi

  chown root:root /etc/init.d/${INSTANCE_INIT_NAME}
  chmod 755 /etc/init.d/${INSTANCE_INIT_NAME}

  # register the startup script
  addService ${INSTANCE_INIT_NAME}

else
  echo Not generating init script
fi

if test ${UPGRADE} -gt 0 ; then
  TOMCAT_VERSION=$(cat $TOMCAT_SERVER_HOME/conf/tomcat.version)
  if [ -e ${INSTANCE_BASE}/${INSTANCE_NAME}/bin/tcruntime-ctl.sh ]; then
    if [ $( grep -c "com.springsource.tcserver" "${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties" ) -gt 0 ]; then
      echo migration from TC Server to Tomcat required
      TOMCAT_MIGRATION=1
    fi
  fi

  # Major version is always 8, review code below when moving to 9
  TOMCAT_VERSIONS=(${TOMCAT_VERSION//./ });
  TOMCAT_MAJOR_VERSION=${TOMCAT_VERSIONS[0]}
  TOMCAT_MINOR_VERSION=${TOMCAT_VERSIONS[1]}

  rm -f $FLAG_TOMCAT_MIGRATION

  CURRENT_FULL_VERSION=$(cat ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/tomcat.version)
  CURRENT_MAJOR_VERSION=${CURRENT_FULL_VERSION%%.*}

  if [[ -n "$TOMCAT_VERSION" && "$CURRENT_FULL_VERSION" == "$TOMCAT_VERSION" ]]; then
    echo Upgrade not required
  else
    SERVER_XML=${INSTANCE_BASE}/${INSTANCE_NAME}/conf/server.xml

    if [[ $(getNewTemplate) = "nio-ssl" && $(grep -c Http11Protocol ${SERVER_XML}) -gt 0 ]]; then
      echo Upgrade the connector type from BIO to NIO
      sed -i -e "s/bio-ssl/nio-ssl/g" -e "s/Http11Protocol/Http11Nio2Protocol/g" $SERVER_XML
      sed -i -e "s/bio-ssl/nio-ssl/g" ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/catalina.properties
    fi

    if [[ -n "$CURRENT_MAJOR_VERSION" && (($CURRENT_MAJOR_VERSION -lt $TOMCAT_MAJOR_VERSION) || $TOMCAT_MIGRATION -gt 0) ]]; then
      echo Performing migration upgrade to ${TOMCAT_VERSION}

      # Stop the instance for migration
      isRunning=`service ${INSTANCE_INIT_NAME} status | grep -c "RUNNING as"`

      if [ $isRunning -gt 0 ]; then
        service ${INSTANCE_INIT_NAME} stop
      fi

      echo $isRunning > $FLAG_TOMCAT_MIGRATION

      MIGRATE_INSTANCE_DIR="${INSTANCE_BASE}/${INSTANCE_NAME}.migrate"
      mv ${INSTANCE_BASE}/${INSTANCE_NAME} ${MIGRATE_INSTANCE_DIR}

      args=( $@ )
      remove_args=( "-U" "-T" )

      for arg in ${remove_args[@]}; do
         args=(${args[@]/$arg})
      done

      /usr/local/horizon/scripts/installTcServer.sh "${args[@]}"

      MIGRATION_FILES=( conf/catalina.properties
                        conf/server.xml )

      echo Saving configuration files:
      for f in ${MIGRATION_FILES[@]} ; do
        TO_MIGRATE=${MIGRATE_INSTANCE_DIR}/${f}
        if test -e $TO_MIGRATE ; then
          cp -v -R -f ${TO_MIGRATE} ${INSTANCE_BASE}/${INSTANCE_NAME}/${f}.migration
        fi
      done

      MIGRATE_FILES=( conf/tcserver.keystore
                      conf/clienttrust.keystore
                      endorsed )

      echo Migrating configuration files:
      for f in ${MIGRATE_FILES[@]} ; do
        TO_MIGRATE=${MIGRATE_INSTANCE_DIR}/${f}
        if test -e $TO_MIGRATE ; then
          cp -v -R -f ${TO_MIGRATE} ${INSTANCE_BASE}/${INSTANCE_NAME}/${f}
        fi
      done

      # set ownership of instance
      /bin/chown -R $TOMCAT_USER:$TOMCAT_GROUP $INSTANCE_BASE

      rm -fr ${MIGRATE_INSTANCE_DIR}

      if [ -e ${INSTANCE_BASE}/${INSTANCE_NAME}/bin/tcruntime-ctl.sh ]; then
        rm -f ${INSTANCE_BASE}/${INSTANCE_NAME}/bin/tcruntime-ctl.sh
      fi

    else
      # This section is executed when there is change in minor version of Tomcat Server.
      # As of now, changes to any config files is not required as Tomcat doesn't have any upgrade script for change in Minor version and existing configs would work as is.
      # Just needs restart of Tomcat service, as Tomcat server is upgraded to newer minor version.
      # This code is to upgrade tomcat.version file while upgrading minor version of Tomcat to Tomcat
      TOMCAT_EXECUTION_FILE=${TOMCAT_INSTANCE}/bin/tomcat-instance-control.sh
      TOMCAT_SERVER_HOME=/usr/share/tomcat
      TOMCAT_VERSION=$(cat ${TOMCAT_SERVER_HOME}/conf/tomcat.version)
      CURRENT_FULL_VERSION=$(cat ${TOMCAT_INSTANCE}/conf/tomcat.version)

      if [[ -e ${TOMCAT_EXECUTION_FILE} && $(grep -c '/usr/share/tomcat' $TOMCAT_EXECUTION_FILE) -gt 0 && $TOMCAT_VERSION != $CURRENT_FULL_VERSION ]] ; then
        echo "${TOMCAT_VERSION}" > ${TOMCAT_INSTANCE}/conf/tomcat.version
      fi
      echo Upgrading tomcat server instance to ${TOMCAT_VERSION}
    fi
    # fix permissions modified by the upgrade
    find ${INSTANCE_BASE}/${INSTANCE_NAME}/webapps -type l -exec chmod 644 "{}" \;
    for f in $(find ${INSTANCE_BASE}/${INSTANCE_NAME}/webapps -type l -printf %l); do
      if [[ $f == /etc/ssl/certs/* ]]; then
        chmod 644 $f;
     fi
    done
  fi
  echo "${TOMCAT_VERSION}" > ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/tomcat.version
  chown ${TOMCAT_USER}:${TOMCAT_GROUP} ${INSTANCE_BASE}/${INSTANCE_NAME}/conf/tomcat.version
fi
