#!/bin/bash
# This script handles missing group IDs in /etc/group
set -e

trap Clean_up HUP INT QUIT TERM ERR EXIT

SCRIPT_NAME=$(basename "${0%.sh}")

PASSWD=/etc/passwd
ETCGROUP=/etc/group

TMPFILE=`mktemp`

# Get all unique GIDs from the /etc/passwd file
cat $PASSWD | awk -F":" '{print $4}' | sort | uniq >> $TMPFILE

while read GID
do
   # Missing group ID in groups file?
   if [ `grep -c ":$GID:" $ETCGROUP` -eq 0 ]; then

      # Orphan users: Users belonging to a group whose GID is not in /etc/group

      # We have to choose a group name for this missing group
      # Candidate group name: Either
      # 1) A group name that EXISTS in /etc/group that's the same name as one
      # of the orphan users, but doesn't have any users with its GID. In other
      # words, its GID is wrong. We set it to the GID we're currently
      # examining.
      # 2) A group name that *doesn't* exist in /etc/group. Choose an arbitrary
      # name from the orphan user list that matches this criteria.

      # Find orphan users of this group
      USERS=
      while read ENTRY
      do
         if [ `echo "$ENTRY" | awk -F":" '{print $4}'` -eq $GID ]; then
            orphan_user=`echo "$ENTRY" | awk -F":" '{print $1}'`
            USERS="$USERS $orphan_user"
         fi
      done < $PASSWD

      CANDIDATE_GROUP=
      EXISTING=0
      for user in $USERS
      do
         if [ `grep -c "^$user:" $ETCGROUP` -gt 0 ]; then
            TEMPGID=`grep "^$user:" $ETCGROUP | awk -F":" '{print $3}'`
            # Now find out if there are any users with this GID. If so, this
            # group may be an unrelated but valid group with the same name as
            # $user
            if [ `awk -F":" '{print $4}' $PASSWD | grep -c "$TEMPGID"` -gt 0 ]
            then
               continue
            fi
            # Else, this group probably has the wrong GID
            CANDIDATE_GROUP="$user"
            EXISTING=1
            break
         else
            CANDIDATE_GROUP="$user"
         fi
      done

      # Found a candidate group? If not, flag an error
      if [ -z "$CANDIDATE_GROUP" ]; then
         echo "WARNING:$SCRIPT_NAME: No candidate group found for $GID, resolve manually"
         continue
         #exit -1
      fi

      # Existing group? Modify it
      if [ $EXISTING -eq 1 ]; then
         OLDGID=`grep "^$CANDIDATE_GROUP:" $ETCGROUP | awk -F":" '{print $3}'`
         groupmod -g $GID "$CANDIDATE_GROUP" > /dev/null 2>&1
         # Sweep across the filesystem and update GIDs
         find / -xdev -gid $OLDGID -exec chgrp $GID {} > /dev/null 2>&1 \;
      else
         groupadd -g $GID "$CANDIDATE_GROUP" > /dev/null 2>&1
      fi

      # Add all candidate users to this group
      for user in $USERS
      do
         gpasswd -a $user $CANDIDATE_GROUP > /dev/null 2>&1 || :
      done
   fi
done < $TMPFILE
