#!/bin/bash
# This script is the top-level script that runs the individual PDI correcting
# scripts

Usage()
{
   echo ""
   echo "Usage: $0 [-h] [-f listFile] [-l lockfileDir]"
   echo ""
   echo "listFile contains the file names of the PDI-correcting scripts, one
   per line"
   echo "lockfileDir specifies the dir which will contain the lock file"
}

Release_Lock()
{
   exit_status=${1:-$?}
   if [ -n "$LOCKFILE" ]; then
      rm -f "$LOCKFILE"
   fi
   exit $exit_status
}

# Change to the dir where this script resides
cd `dirname $0`

# Source the common variables and functions
. sourcedVars
. Functions

SCRIPT_NAME=$(basename "${0%.sh}")

# Make sure we are root
if [ ! IsRoot ]; then
   echo "$SCRIPT_NAME:ERROR: This script must be run as root."  >&2
   exit 1
fi

trap "{ (Clean_up); Release_Lock; }" HUP INT QUIT TERM EXIT

# Prepare the list of PDIs to be executed
TMPFILE=`mktemp`
if [ -e OverriddenPDIs ]; then
   grep -v "^#" OverriddenPDIs | sed "s/^.*$/&.sh/g" > "$TMPFILE"
fi
EXEC_LIST=`ls GEN* LNX* VMW* CVE* NSA* | sort -d - "$TMPFILE" | uniq -u`
LOCKDIR=/tmp

while getopts ":f:l:h" opt; do
   case $opt in
      f)
      EXEC_LIST=`cat $OPTARG`
      ;;
      l)
      LOCKDIR=$OPTARG
      ;;
      h)
      Usage
      exit 0
      ;;
      \?)
      echo "Invalid option $opt"
      Usage
      exit 1
      ;;
   esac
done

STIGREPORT=/var/log/stigreport.log
touch $STIGREPORT
chown root:root $STIGREPORT
chmod 600 $STIGREPORT

# Paranoia: Avoid simultaneous executions from racing
LOCKFILE=$LOCKDIR/.srrlock
if ( set -o noclobber ; echo "foo" > $LOCKFILE ) 2> /dev/null; then
   # --BEGIN critical section--
   # Run the delete users script first to avoid misleading results
   ./DeleteUsers.sh

   # Now run the individual PDIs
   for PDI in $EXEC_LIST
   do
      if [ -x $PDI ]; then
         [ ${PDI:0:1} == "/" ] && CMD=$PDI || CMD=$PWD/$PDI
         /bin/bash $CMD >> $STIGREPORT 2>&1
         echo "$PDI STIG applied" >> $STIGREPORT
      else
         echo "$PDI STIG skip" >> $STIGREPORT
      fi
   done

   # Run the ChmodFiles script last to make sure we override any perm changes
   # made by individual scripts

   # Prepare the list of files to be chmod'ed
   ./GenChmodList.sh | cat - StaticChmodList > "$TMPFILE"
   if [ -e OverriddenPDIs ]; then
      # Filter the Chmod list through the override PDI list
      grep -v "^#" "$TMPFILE" | while read LINE
      do
         PDI=`echo "$LINE" | awk -F":[[:space:]]*" '{ print $3 }'`
         if [ `grep -c "^$PDI$" OverriddenPDIs` -eq 0 ]; then
            echo "$LINE" >> ChmodList
         fi
      done
   else
      mv "$TMPFILE" ChmodList
   fi

   ./ChmodFiles.sh
   rm -f ChmodList
   # --END critical section--
else
    echo "$SCRIPT_NAME:ERROR: Could not acquire lock file" >&2
    exit -1
fi
