#!/bin/sh

INSTANCE_BASE=${INSTANCE_BASE:-/opt/vmware/horizon}
INSTANCE_NAME=workspace
INSTANCE_KEYSTORE=$INSTANCE_BASE/$INSTANCE_NAME/conf/tcserver.keystore
INSTANCE_SETENV=$INSTANCE_BASE/$INSTANCE_NAME/bin/setenv.sh
INSTANCE_SERVICE=horizon-workspace
TOMCAT_USER=${TOMCAT_USER:-horizon}
TOMCAT_GROUP=${TOMCAT_GROUP:-www}
TOMCAT_INSTANCE=$INSTANCE_BASE/$INSTANCE_NAME
TC_CONF_CONTEXT=$INSTANCE_BASE/$INSTANCE_NAME/conf/context.xml
TC_CONF_FILE=$INSTANCE_BASE/$INSTANCE_NAME/conf/server.xml

SCRIPT_PATH=${SCRIPT_PATH:-/usr/local/horizon/scripts}
JRE_ENDORSED_PATH=${JRE_ENDORSED_PATH:-/usr/local/horizon/jre-endorsed}

. ${SCRIPT_PATH}/utils.inc
. ${SCRIPT_PATH}/flag.inc

# Add a Connector for the given port as a copy of the default https connector
# addHttpsConnector <new port> [<new ssl alias>]
addHttpsConnector(){
    copyConnector '\${nio-ssl.https.port}' $1 $1 $2
}

# Add a Connector for the given port as a copy of the specified port connector
copyConnector(){
    local FROM_PORT=$1
    local TO_PORT=$2
    local REDIRECT_PORT=$3
    local ALIAS=${4-'\1'}

    if test `grep -c port=\"${TO_PORT}\" $TC_CONF_FILE` -eq 0 ; then
        NEW_CONNECTOR=`awk '/<Connector/,/<\/Connector>/ {
            if ( !($0 ~ /address=/) ){
                connector=connector $0"_NEWLINE_";
            }
            if (/<\/Connector>/){
                if(connector ~ /port="'"${FROM_PORT}"'"/){
                    printf "%s", connector;
                    exit;
                }
                connector="";
            }
        }' ${TC_CONF_FILE} | sed ${SED_FLAG} -e 's/port="[^"]+"/port="'"${TO_PORT}"'"/g' \
                                             -e 's/certificateKeyAlias="([^"]+)"/certificateKeyAlias="'"${ALIAS}"'"/g' \
                                             -e 's/redirectPort="[^"]+"/redirectPort="'"${REDIRECT_PORT}"'"/g' `

        cp ${TC_CONF_FILE} ${TC_CONF_FILE}.bk

        sed '1,/<\/Service>/ {/<\/Service>/ i\
'"$NEW_CONNECTOR "'
}' ${TC_CONF_FILE}.bk | sed 's/_NEWLINE_/\
/g' | tr -d '\r' > ${TC_CONF_FILE}
    fi
}

setHttpOnly(){
    # set the httponly flag to true
    sed -i.bk 's/<Context>/<Context useHttpOnly="true">/' $TC_CONF_CONTEXT
}

setOwnership(){
    # set ownership of instance
    chown -R $TOMCAT_USER:$TOMCAT_GROUP $INSTANCE_BASE
    find $INSTANCE_BASE -type d -exec chmod 750 "{}" \;
}

getTomcatVersion(){
    VERSION=${INSTANCE_BASE}/${INSTANCE_NAME}/conf/tomcat.version
    echo ${VERSION:-8.5.0}
}

getTomcatMajorVersion(){
    local version=$(getTomcatVersion)
    echo ${version%%.*}
}

deployWar(){
    local warFile=$1
    local appName=$2
    local appDir=$INSTANCE_BASE/$INSTANCE_NAME/${3-webapps}/${appName}
    local workDir=${4-Catalina}
    local workHost=${5-localhost}
    local contextFile=${appDir}/META-INF/context.xml
    local majorVersion=$(getTomcatMajorVersion)

    if [ ! -f $warFile ]; then return; fi

    rm -rf ${appDir}
    unzip ${warFile} -d ${appDir} &> /tmp/exploded_${appName}_war.log

    rm -fr $INSTANCE_BASE/$INSTANCE_NAME/work/${workDir}/${workHost}/${appName}
    rm -fr $INSTANCE_BASE/$INSTANCE_NAME/temp/*

    #if there is a specific context file for this version of tomcat, use it
    if [ -f ${contextFile}.${majorVersion} ] ; then
        cp ${contextFile} ${contextFile}.orig
        cp ${contextFile}.${majorVersion} ${contextFile}
    fi

    #if running in FIPS mode, remove bc-fips from deployed war
    if [ -f $FLAG_FIPS_MODE ] ; then
        rm -f ${appDir}/WEB-INF/lib/bc-fips*jar
    fi

    chown -R $TOMCAT_USER:$TOMCAT_GROUP ${appDir}
}

deploySecureWar(){
  deployWar $1 $2 secure-webapps SecureCatalina secure-localhost
}

removeWar(){
    local appName=$1
    local appDir=${2-webapps}
    /bin/rm -rf $INSTANCE_BASE/$INSTANCE_NAME/${appDir}/${appName}
}

removeSecureWar(){
    removeWar $1 secure-webapps
}

#Handle lines with continuation characters
updateInstanceProperty(){
  local KEY=$1
  local VAL=$2
  replacePropertyWithContinuation "${KEY}" "${VAL}" ${TOMCAT_INSTANCE}/conf/catalina.properties
}

removeInstanceProperty(){
  local KEY=$1
  removeProperty "${KEY}" ${TOMCAT_INSTANCE}/conf/catalina.properties
}

getInstanceProperty(){
    getPropertyVal "$1" ${TOMCAT_INSTANCE}/conf/catalina.properties
}

getTemplateName(){
    local protocol=$(awk -F'"' '/protocol=/ {print $2; exit}' ${TC_CONF_FILE})

    case "$protocol" in
        org.apache.coyote.http11.Http11Protocol) echo bio-ssl;;
        org.apache.coyote.http11.Http11NioProtocol) echo nio-ssl;;
        org.apache.coyote.http11.Http11Nio2Protocol) echo nio-ssl;;
        org.apache.coyote.http11.Http11AprProtocol) echo apr-ssl;;
        *) echo nio-ssl;;
    esac
}

getInstanceKeystorePwd(){
    getInstanceProperty $(getTemplateName).ssl.keystore.password
}

getInstanceSSLAlias(){
    getInstanceProperty $(getTemplateName).ssl.key.alias
}

getPassthroughSSLAlias(){
    getInstanceProperty ssl.passthrough.alias
}

getPassthroughPort(){
     getInstanceProperty https.passthrough.port
 }

getPassthroughPortProperty(){
     echo https.passthrough.port
 }

getHTTPSPort() {
    getInstanceProperty nio-ssl.https.port
}

getInstanceKeystoreType(){
    getInstanceProperty $(getTemplateName).ssl.keystore.type
}

getInstanceClientTrustKeystorePwd(){
    getInstanceProperty les.keystore.password
}

getInstanceClientTrustKeystore(){
    getInstanceProperty les.keystore.location
}

getInstanceClientTrustKeystoreType(){
    getInstanceProperty les.keystore.type
}

# Tell whether or not a given webapp is deployed on this tomcat
# instance.
# @param appName the webapp context name as deployed
# @param appDir  optional webapps directory, default is 'webapps'
# @return 1 if deployed, 0 otherwise.
isAppDeployed(){
    local appName=$1
    local appDir=${2:-webapps}
    if [ -d $INSTANCE_BASE/$INSTANCE_NAME/${appDir}/${appName} ]; then
      echo 1
    else
      echo 0
    fi
}

getAppVersion(){
    local appName=$1
    local appDir=${2:-webapps}
    local manifest=$INSTANCE_BASE/$INSTANCE_NAME/${appDir}/${appName}/META-INF/MANIFEST.MF
    local version="unknown"
    if [ -f $manifest ]; then
      version=$(getPropertyVal "Implementation-Version" "${manifest}" ":")
    fi
    echo $version
}

removeOldSmartCardAuthConfig() {
    CLIENT_TRUST_KEYSTORE=$(getInstanceProperty smartcard.keystore.location)

    if test -n "$CLIENT_TRUST_KEYSTORE" ; then
      rm -f "$CLIENT_TRUST_KEYSTORE"
    fi

    removeInstanceProperty smartcard.keystore.password
    removeInstanceProperty smartcard.keystore.location
}

enableSmartCardAuth() {
    FIPS_JAR=$1
    if [ -f $FLAG_FIPS_MODE ] ; then
      if [ -z "$FIPS_JAR" ] ; then
        FIPS_JAR=$(ls -1r ${JRE_ENDORSED_PATH}/bc-fips-*.jar | head -1)
      fi
    fi

    removeOldSmartCardAuthConfig

    if test $(grep -c "honorCipherOrder" ${TC_CONF_FILE}) -gt 0; then
        sed -i.bk -r -e '/honorCipherOrder/ a\
                           trustManagerClassName="com.vmware.horizon.tomcat.CustomTrustManager"\
                           certificateVerification="false"'\
              -e "/^[[:space:]]*certificateVerification[[:space:]]*=/d"\
              -e "/^[[:space:]]*trustManagerClassName[[:space:]]*=/d"\
              -e "/^[[:space:]]*truststoreFile[[:space:]]*=/d"\
              -e "/^[[:space:]]*truststorePassword[[:space:]]*=/d"\
              -e "/^[[:space:]]*truststoreProvider[[:space:]]*=/d"\
              -e "/^[[:space:]]*truststoreType[[:space:]]*=/d"\
              ${TC_CONF_FILE}
    fi
}
