-- This script is to create an initial vPostgres database schema for UM 3.6.0

-- Attention: this file should be synced with 350-360.sql:
-- when a change is made here, the other should be also updated. The final schemas should be the same.

create table "CollectionSettings" (
    "id" integer not null,
    "startMinute" integer not null
  );
create table "Collection" (
    "id" integer primary key not null,
    "start" timestamp not null
  );
create sequence "s_Collection_id";
-- indexes on Collection
create index "collection_start" on "Collection" ("start");

create table "CollectionMsg" (
    "id" integer primary key not null,
    "collectionId" integer not null,
    "time" timestamp,
    "serverTypeId" integer,
    "host" varchar(255),
    "message" varchar(10000) not null
  );
create sequence "s_CollectionMsg_id";
create table "VcCollectionMsg" (
    "id" integer primary key not null,
    "firstTime" timestamp not null,
    "lastTime" timestamp not null,
    "occurs" integer not null,
    "host" varchar(255) not null,
    "message" varchar(10000) not null
  );
create sequence "s_VcCollectionMsg_id";
-- indexes on VcCollectionMsg
create index "VcCollectionMsg_firstTime" on "VcCollectionMsg" ("firstTime");

create table "Customer" (
    "id" integer primary key not null,
    "providerId" integer not null,
    "name" varchar(128) not null
  );
create sequence "s_Customer_id";
-- indexes on Customer
create unique index "idx24220521" on "Customer" ("name");
create table "CustomerLocation" (
    "id" integer primary key not null,
    "customerId" integer not null,
    "countryCode" varchar(128) not null,
    "postalCode" varchar(128) not null
  );
create sequence "s_CustomerLocation_id";
create table "CustomerVmRule" (
    "id" integer primary key not null,
    "customerId" integer not null,
    "vcServerId" integer,
    "objectType" integer not null,
    "valueType" integer not null,
    "value" varchar(128),
    "createdBy" integer not null
  );
create sequence "s_CustomerVmRule_id";
create table "EmailAlerts" (
    "id" integer primary key not null,
    "to" varchar(128) not null,
    "from" varchar(128) not null,
    "okCollection" boolean not null,
    "failedCollection" boolean not null
  );
create sequence "s_EmailAlerts_id";
create table "ReportSettings" (
    "id" integer primary key not null,
    "perVmMemCapMb" integer not null
  );
create table "ReportSet" (
    "id" integer primary key not null,
    "name" varchar(128) not null,
    "reportingDay" integer not null,
    "toVsppPortal" boolean not null default(false),
    "to" varchar(128) not null,
    "from" varchar(128) not null,
    "lastSent" timestamp
  );
create sequence "s_ReportSet_id";
create table "EmailServer" (
    "id" integer primary key not null,
    "server" varchar(128),
    "port" integer not null,
    "user" varchar(128),
    "password" varchar(128),
    "connectionSecurity" integer not null
  );
create sequence "s_EmailServer_id";
create table "ReportSetReport" (
    "reportSetId" integer not null,
    "reportId" integer not null
  );
create table "ReportSetLicenseSet" (
    "reportSetId" integer not null,
    "licenseSetId" integer not null
  );
create table "Host" (
    "id" integer primary key not null,
    "uuid" varchar(128) not null,
    "vcServerId" integer not null,
    "moref" varchar(128) not null
  );
create sequence "s_Host_id";
create table "HostHistory" (
    "id" integer primary key not null,
    "hostId" integer not null,
    "time" timestamp not null,
    "nextTime" timestamp not null,
    "fullName" varchar(128) not null,
    "version" varchar(128) not null,
    "name" varchar(128) not null,
    "licenseId" integer not null,
    "memSize" bigint not null,
    "licenseCategoryId" integer not null default(1),
    "vcd" boolean not null default(false)
  );
create sequence "s_HostHistory_id";
create table "InventoryItem" (
    "id" integer primary key not null,
    "vcServerId" integer not null,
    "nodeType" integer not null,
    "moref" varchar(128) not null,
    "name" varchar(128) not null,
    "parentMoref" varchar(128) not null,
    "resourcePoolMoref" varchar(128),
    "discovered" timestamp not null,
    "removed" timestamp
  );
create sequence "s_InventoryItem_id";
create table "LdapServer" (
    "id" integer not null,
    "host" varchar(128) not null,
    "port" integer not null,
    "useSsl" boolean not null,
    "dn" varchar(256) not null,
    "password" varchar(128) not null,
    "baseDn" varchar(256) not null,
    "objectClass" varchar(128) not null,
    "userNameAttr" varchar(128) not null
  );
create table "License" (
    "id" integer primary key not null,
    "name" varchar(128) not null,
    "code" varchar(128) not null,
    "licenseCategoryId" integer not null
  );
create sequence "s_License_id";
-- indexes on License
create unique index "idx1d64048c" on "License" ("code");
create table "LicenseCategory" (
    "id" integer primary key not null,
    "name" varchar(128) not null,
    "billable" boolean not null
  );
create sequence "s_LicenseCategory_id";
-- indexes on LicenseCategory
create unique index "idx518407d0" on "LicenseCategory" ("name");
create table "LicenseSet" (
    "id" integer primary key not null,
    "name" varchar(128) not null,
    "addNewHere" boolean not null default(false)
  );
create sequence "s_LicenseSet_id";
-- indexes on LicenseSet
create unique index "idx2dd705be" on "LicenseSet" ("name");
create table "LicenseSetLicense" (
    "licenseSetId" bigint not null,
    "licenseId" bigint not null
  );
create table "Provider" (
    "id" integer primary key not null,
    "company" varchar(128) not null,
    "contact" varchar(128) not null,
    "phone" varchar(128) not null,
    "email" varchar(128) not null,
    "partnerId" varchar(128) not null,
    "contractNum" varchar(128) not null,
    "siteId" varchar(128) not null,
    "portalUserName" varchar(128) not null,
    "portalPassword" varchar(128) not null
  );
create sequence "s_Provider_id";
create table "SchemaInfo" (
    "id" integer not null,
    "version" integer not null,
    "migrationDone" boolean not null
  );
create table "User" (
    "role" varchar(100) not null,
    "password" varchar(100) not null,
    "salt" varchar(100) not null,
	"username" varchar(100) not null
  );
create table "VcacServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "host" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
    "cafeHost" varchar(128) not null,
    "solutionUser" varchar(128) not null,
    "licenseNameId" integer not null
  );
create sequence "s_VcacServer_id";
-- indexes on VcacServer
create unique index "VcacServer_host" on "VcacServer" ("host");
create table "VcacMonth" (
    "id" integer primary key not null,
    "vcacServerId" integer not null,
    "year" integer not null,
    "month" integer not null,
    "maxValue" integer not null
  );
create sequence "s_VcacMonth_id";
create table "VcacLicenseName" (
    "id" integer primary key not null,
    "name" varchar(128) not null
  );
create sequence "s_VcacLicenseName_id";
create table "VcdServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "version" varchar(128) not null,
    "host" varchar(128) not null,
    "port" integer not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
	"serialNumber" varchar(128) not null
  );
create sequence "s_VcdServer_id";
create table "VcdServerCustomer" (
    "vcdServerId" integer not null,
    "customerId" integer not null
  );
create table "VcdVdcInfo" (
    "id" integer primary key not null,
    "vcdServerId" integer not null,
    "time" timestamp not null,
    "orgName" varchar(128) not null,
    "orgId" varchar(128) not null,
    "vdcName" varchar(128) not null,
    "vdcId" varchar(128) not null,
    "numVapps" integer not null,
    "numVms" integer not null
  );
create sequence "s_VcdVdcInfo_id";
create table "VcopsServer" (
    "id" integer primary key not null,
    "host" varchar(128) not null,
    "version" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
    "active" boolean not null,
	"licenseNameId" integer not null
  );
create sequence "s_VcopsServer_id";
-- indexes on VcopsServer
create unique index "VcopsServerHost" on "VcopsServer" ("host");
create table "NsxServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "host" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
    "port" integer not null DEFAULT 443,
    "vchost" varchar(128) not null
  );
create sequence "s_NsxServer_id";
-- indexes on NsxServer
create unique index "NsxServerHost" on "NsxServer" ("host");
create table "VcServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "host" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
	"instanceUuid" varchar(128) not null,
    "fullName" varchar(128) not null,
    "version" varchar(128) not null,
    "meter" boolean not null,
    "port" integer not null,
	"vcdServerId" integer not null default(0),
	"sso" integer DEFAULT 0,
	"pscId" integer
  );
create sequence "s_VcServer_id";
-- indexes on VcServer
create unique index "VcServer_instanceUuid" on "VcServer" ("instanceUuid");
create table "VcVcops" (
    "vcServerId" bigint not null,
    "vcopsServerId" bigint not null,
    "changed" timestamp not null
  );
create table "Vm" (
    "id" integer primary key not null,
    "instanceUuid" varchar(128) not null,
    "moref" varchar(128) not null,
    "vcServerId" integer not null,
    "vcName" varchar(128) not null,
    "guestHostName" varchar(128),
    "ipV4Address" varchar(128),
    "reportExclusionReason" integer,
	"biosUuid" varchar(128) not null,
	"nsxUsageLicense" integer not null default(0),
    "mac" varchar(128)
  );
create sequence "s_Vm_id";
create table "VmHistory" (
    "id" integer primary key not null,
    "vmId" integer not null,
    "time" timestamp not null,
    "nextTime" timestamp not null,
    "hostId" integer not null,
    "connState" integer not null,
    "powerState" integer not null,
    "vmMemSize" integer not null,
    "vmMemRes" integer not null,
    "guestFullName" varchar(1000),
    "cpuCount" integer not null,
    "vrops" boolean not null,
	"nsxUsageLicense" integer not null default(0)
  );
create sequence "s_VmHistory_id";
create table "ProtectedUuidMonth" (
    "id" integer primary key not null,
    "name" varchar(128) not null,
    "biosUuid" varchar(128) not null,
    "year" integer not null,
    "month" integer not null
  );
create sequence "s_ProtectedUuidMonth_id";
create table "CustomerVm" (
    "id" integer primary key not null,
    "customerId" integer not null,
    "vmId" integer not null,
    "customerVmRuleId" integer not null,
    "changed" timestamp not null
  );
create sequence "s_CustomerVm_id";
-- indexes on CustomerVm
create unique index "idx2fab05d3" on "CustomerVm" ("vmId");
create table "CustomerCluster" (
    "id" integer primary key not null,
    "customerId" integer not null,
    "clusterId" integer not null,
    "customerVmRuleId" integer not null,
    "changed" timestamp not null
  );
create sequence "s_CustomerCluster_id";
create unique index "customeCluster_clusterId" on "CustomerCluster" ("clusterId");
create table "Token" (
    "id" integer primary key not null,
    "role" varchar(128) not null,
    "value" varchar(128) not null
    );
create sequence "s_Token_id";
create table "VcEnv" (
    "id" integer primary key not null,
    "vcServerId" integer not null,
    "envCode" integer not null
  );
create sequence "s_VcEnv_id";
-- indexes on VcEnv
create unique index "vcEnv_vcServerId" on "VcEnv" ("vcServerId");
create table "VcopsVmCollection" (
    "id" integer primary key not null,
    "vmId" integer not null,
    "timestamp" timestamp not null,
    "vcopsLicenseNameId" integer not null
  );
create sequence "s_VcopsVmCollection_id";
-- indexes on VcopsVmCollection
create index "VcopsVmCollection_timestamp" on "VcopsVmCollection" ("timestamp");
create table "VcopsLicenseName" (
    "id" integer primary key not null,
    "name" varchar(128) not null
  );
create sequence "s_VcopsLicenseName_id";

create table "SrmVcPair" (
    "id" integer primary key not null,
    "vcServerId1" integer not null,
    "vcServerId2" integer not null
  );
create sequence "s_SrmVcPair_id";

create unique index "SrmVcPair_vcServerId1" on "SrmVcPair" ("vcServerId1");

create table "VcSrm" (
    "id" integer primary key not null,
    "vcServerId" integer not null,
    "srmHostnames" varchar(512) not null default(''),
    "updating"   boolean not null default(false)
  );
create sequence "s_VcSrm_id";
-- indexes on VcSrm
create unique index "VcSrm_vcServerId" on "VcSrm" ("vcServerId");

create table "DesktoneServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "host" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
	"sessionModels" integer not null default(0)
  );
create sequence "s_DesktoneServer_id";
-- indexes on DesktoneServer
create unique index "DesktoneServer_host" on "DesktoneServer" ("host");
create table "DesktoneMonth" (
    "id" integer primary key not null,
    "desktoneServerId" integer not null,
    "year" integer not null,
    "month" integer not null,
    "desktopQuota" integer not null,
	"sessionQuota" integer not null,
	"tenant" varchar(1000)
  );
create sequence "s_DesktoneMonth_id";
-- Column "batch": 1, old bundles; 2, new bundles starting July 1, 2016
create table "Bundle" (
    "id" integer not null,
    "name" varchar(128) not null,
    "points" float not null,
    "batch" integer not null
  );
create table "Product" (
    "id" integer not null,
    "name" varchar(128) not null,
    "separable" boolean not null
  );
create table "ProductInBundle" (
    "bundleId" integer not null,
    "productId" integer not null
  );
create table "SeparateProduct" (
    "productId" integer not null
  );

create table "ClusterVsanHistory" (
    "id" integer primary key not null,
    "vcServerId" integer not null,
    "clusterId" integer not null,
    "vsanLicCategory" integer not null,
    "usedTotalMb" bigint not null,
    "deDupEnabled" boolean not null default(false),
    "erasureCoding" boolean not null default(false),
    "stretchedClusterEnabled" boolean not null default(false),
    "iopsLimitUsed" boolean not null default(false),
    "startTime" timestamp not null,
    "endTime" timestamp not null
  );
create sequence "s_ClusterVsanHistory_id";

create table "Cluster" (
    "id"   integer primary key not null,
    "clusName" varchar(128) not null,
    "dataStoreName" varchar(128) not null,
    "vcServerId" integer not null,
    "vsanEnabled" boolean not null,
    "lastChanged" timestamp not null
);
create sequence "s_Cluster_id";

CREATE SEQUENCE "s_PscServer_id";
CREATE TABLE "PscServer" (
"id"   integer PRIMARY KEY NOT NULL,
"host" varchar(128) NOT NULL,
"port" integer DEFAULT 7444 NOT NULL,
"createTime" timestamp NOT NULL
);

-- foreign key constraints :
alter table "CollectionMsg" add constraint "CollectionMsgFK1" foreign key ("collectionId") references "Collection"("id");
alter table "CustomerLocation" add constraint "CustomerLocationFK1" foreign key ("customerId") references "Customer"("id");
alter table "HostHistory" add constraint "HostHistoryFK2" foreign key ("hostId") references "Host"("id");
alter table "HostHistory" add constraint "HostHistoryFK3" foreign key ("licenseId") references "License"("id");
alter table "License" add constraint "LicenseFK4" foreign key ("licenseCategoryId") references "LicenseCategory"("id");
alter table "VcacServer" add constraint "VcacServerFK7" foreign key ("licenseNameId") references "VcacLicenseName"("id");
alter table "VcacMonth" add constraint "VcacMonthFK7" foreign key ("vcacServerId") references "VcacServer"("id");
alter table "DesktoneMonth" add constraint "DesktoneMonthFK8" foreign key ("desktoneServerId") references "DesktoneServer"("id");
alter table "VcdVdcInfo" add constraint "VcdVdcInfoFK11" foreign key ("vcdServerId") references "VcdServer"("id");
alter table "VmHistory" add constraint "VmHistoryFK14" foreign key ("hostId") references "Host"("id");
alter table "VmHistory" add constraint "VmHistoryFK15" foreign key ("vmId") references "Vm"("id");
alter table "CustomerVm" add constraint "CustomerVmFK16" foreign key ("customerId") references "Customer"("id");
alter table "CustomerVmRule" add constraint "CustomerVmRuleFK17" foreign key ("customerId") references "Customer"("id");
alter table "CustomerVm" add constraint "CustomerVmFK18" foreign key ("vmId") references "Vm"("id");
alter table "CustomerVm" add constraint "CustomerVmFK19" foreign key ("customerVmRuleId") references "CustomerVmRule"("id");
alter table "ReportSetReport" add constraint "ReportSetReportFK20" foreign key ("reportSetId") references "ReportSet"("id");
alter table "VcopsVmCollection" add constraint "VcopsVmCollectionFK23" foreign key ("vmId") references "Vm"("id");
alter table "VcopsVmCollection" add constraint "VcopsVmCollectionFK24" foreign key ("vcopsLicenseNameId") references "VcopsLicenseName"("id");
alter table "VcopsServer" add constraint "VcopsServerFK25" foreign key ("licenseNameId") references "VcopsLicenseName"("id");
alter table "Host" add constraint "HostFK25" foreign key ("vcServerId") references "VcServer"("id");
alter table "InventoryItem" add constraint "InventoryItemFK26" foreign key ("vcServerId") references "VcServer"("id");
alter table "SrmVcPair" add constraint "SrmVcPairFK27" foreign key ("vcServerId1") references "VcServer"("id");
alter table "SrmVcPair" add constraint "SrmVcPairFK28" foreign key ("vcServerId2") references "VcServer"("id");
alter table "VcSrm" add constraint "VcSrmFK29" foreign key ("vcServerId") references "VcServer"("id");
alter table "VcEnv" add constraint "VcEnvFK32" foreign key ("vcServerId") references "VcServer"("id");
alter table "CustomerVmRule" add constraint "CustomerVmRuleFK30" foreign key ("vcServerId") references "VcServer"("id");
alter table "Customer" add constraint "CustomerFK31" foreign key ("providerId") references "Provider"("id");
alter table "LicenseSetLicense" add constraint "LicenseSetLicenseFK5" foreign key ("licenseSetId") references "LicenseSet"("id");
alter table "LicenseSetLicense" add constraint "LicenseSetLicenseFK6" foreign key ("licenseId") references "License"("id");
alter table "VcdServerCustomer" add constraint "VcdServerCustomerFK9" foreign key ("vcdServerId") references "VcdServer"("id");
alter table "VcdServerCustomer" add constraint "VcdServerCustomerFK10" foreign key ("customerId") references "Customer"("id");
alter table "VcVcops" add constraint "VcVcopsFK12" foreign key ("vcServerId") references "VcServer"("id");
alter table "VcVcops" add constraint "VcVcopsFK13" foreign key ("vcopsServerId") references "VcopsServer"("id");
alter table "ReportSetLicenseSet" add constraint "ReportSetLicenseSetFK21" foreign key ("reportSetId") references "ReportSet"("id");
alter table "ReportSetLicenseSet" add constraint "ReportSetLicenseSetFK22" foreign key ("licenseSetId") references "LicenseSet"("id");
ALTER TABLE "VcServer" ADD CONSTRAINT "VcServerFK01" FOREIGN KEY ("pscId") REFERENCES "PscServer"("id");
ALTER TABLE "VcServer" ADD CONSTRAINT "VcServerCHK01" CHECK (("sso" <> 2 and "pscId" IS NULL) OR ("sso" = 2 AND "pscId" IS NOT NULL));
--alter table "ClusterVsanHistory" add constraint "ClusterVsanHistoryFK11" foreign key ("inventoryItemId") references "InventoryItem"("id");


-- composite key indexes :
alter table "ReportSetReport" add primary key ("reportSetId","reportId");
alter table "ReportSetLicenseSet" add primary key ("reportSetId","licenseSetId");
alter table "LicenseSetLicense" add primary key ("licenseSetId","licenseId");
alter table "VcdServerCustomer" add primary key ("vcdServerId","customerId");
alter table "VcVcops" add primary key ("vcServerId","vcopsServerId");
-- column group indexes :
create unique index "idx859709f3" on "Host" ("vcServerId","moref","uuid");
create unique index "HostHistory_licenseCategoryId_time_nextTime" on "HostHistory" ("licenseCategoryId", "time", "nextTime", "licenseId", "hostId");
create index "HostHistory_hostId_time" on "HostHistory" ("hostId", "time");
create unique index "InventoryItem_vcServerId_moref" on "InventoryItem" ("vcServerId","moref");
create unique index "VcacMonth_vcacServerId_year_month" on "VcacMonth" ("vcacServerId","year","month");
create unique index "DesktoneMonth_desktoneServerId_year_month_tenant" on "DesktoneMonth" ("desktoneServerId","year","month","tenant");
create unique index "VcdServer_host_port" on "VcdServer" ("host","port");
create unique index "idxc72a0c4d" on "Vm" ("vcServerId","moref","instanceUuid");
create unique index "VmHistory_vmId_time" on "VmHistory" ("vmId","time");
create unique index "VmHistory_vmId_nextTime_vrops" on "VmHistory" ("vmId","nextTime", "vrops");
create index "VmHistory_time_nextTime_states" on "VmHistory" ("time", "nextTime", "connState", "powerState", "hostId");
create unique index "ProtectedUuidMonth_biosUuid_year_month" on "ProtectedUuidMonth" ("biosUuid","year","month");

create index "VcVcops_VcopsServerId" on "VcVcops" ("vcopsServerId");
create index "Vm_vcServerId" on "Vm" ("vcServerId");

ALTER TABLE "VmHistory"   ADD CHECK (time < "nextTime");
ALTER TABLE "HostHistory" ADD CHECK (time < "nextTime");

insert into "LicenseCategory" values(1, 'vCAN', true);
insert into "LicenseCategory" values(2, 'Demo', false);
insert into "LicenseCategory" values(3, 'Perpetual', false);

-- version 16 = 3.6.0. Set migrationDone = true because there's no migration
insert into "SchemaInfo" values(1, 16, true);

insert into "ReportSettings" values(1, 24 * 1024);

insert into "Bundle" values(1, 'Standard Edition',        5, 1);
insert into "Bundle" values(2, 'Premier Edition',         7, 1);
insert into "Bundle" values(3, 'Premier-X Edition',      10, 1);
insert into "Bundle" values(4, 'Premier Plus Edition',   10, 1);
insert into "Bundle" values(5, 'Premier-X Plus Edition', 13, 1);
insert into "Bundle" values(6,  'Starter Bundle',                              3.5, 2);
insert into "Bundle" values(7,  'Standard Bundle',                               5, 2);
insert into "Bundle" values(8,  'Advanced Bundle',                               7, 2);
insert into "Bundle" values(9,  'Standard Bundle with Networking',               8, 2);
insert into "Bundle" values(10, 'Standard Bundle with Management',               8, 2);
insert into "Bundle" values(11, 'Advanced Bundle with Networking',               9, 2);
insert into "Bundle" values(12, 'Advanced Bundle with Management',              10, 2);
insert into "Bundle" values(13, 'Advanced Bundle with Networking & Management', 12, 2);

insert into "Product" values(1,  'vSphere Enterprise',             false);
insert into "Product" values(2,  'vSphere Enterprise Plus',        false);
insert into "Product" values(3,  'vSphere Standard',               false);
insert into "Product" values(4,  'vCenter Server Std',             false);
insert into "Product" values(5,  'NSX Enterprise',                  true);
insert into "Product" values(6,  'NSX Advanced',                    true);
insert into "Product" values(7,  'NSX Base',                        true);
insert into "Product" values(8,  'vRealize Operations Enterprise',  true);
insert into "Product" values(9,  'vRealize Log Insight',            true);
insert into "Product" values(10, 'vCloud Director',                 true);
insert into "Product" values(11, 'vRB for Cloud Adv',               true);
insert into "Product" values(12, 'ChargeBack Manager',              true);

insert into "ProductInBundle" values(1, 1);
insert into "ProductInBundle" values(1, 9);
insert into "ProductInBundle" values(1, 10);
insert into "ProductInBundle" values(1, 11);
insert into "ProductInBundle" values(1, 12);

insert into "ProductInBundle" values(2, 2);
insert into "ProductInBundle" values(2, 9);
insert into "ProductInBundle" values(2, 10);
insert into "ProductInBundle" values(2, 11);
insert into "ProductInBundle" values(2, 12);

insert into "ProductInBundle" values(3, 2);
insert into "ProductInBundle" values(3, 5);
insert into "ProductInBundle" values(3, 6);
insert into "ProductInBundle" values(3, 7);
insert into "ProductInBundle" values(3, 9);
insert into "ProductInBundle" values(3, 10);
insert into "ProductInBundle" values(3, 11);
insert into "ProductInBundle" values(3, 12);

insert into "ProductInBundle" values(4, 2);
insert into "ProductInBundle" values(4, 8);
insert into "ProductInBundle" values(4, 9);
insert into "ProductInBundle" values(4, 10);
insert into "ProductInBundle" values(4, 11);
insert into "ProductInBundle" values(4, 12);

insert into "ProductInBundle" values(5, 2);
insert into "ProductInBundle" values(5, 5);
insert into "ProductInBundle" values(5, 6);
insert into "ProductInBundle" values(5, 7);
insert into "ProductInBundle" values(5, 8);
insert into "ProductInBundle" values(5, 9);
insert into "ProductInBundle" values(5, 10);
insert into "ProductInBundle" values(5, 11);
insert into "ProductInBundle" values(5, 12);

insert into "ProductInBundle" values(6, 4);

insert into "ProductInBundle" values(7, 1);
insert into "ProductInBundle" values(7, 4);
insert into "ProductInBundle" values(7, 10);
insert into "ProductInBundle" values(7, 11);
insert into "ProductInBundle" values(7, 12);

insert into "ProductInBundle" values(8, 2);
insert into "ProductInBundle" values(8, 4);
insert into "ProductInBundle" values(8, 7);
insert into "ProductInBundle" values(8, 9);
insert into "ProductInBundle" values(8, 10);

insert into "ProductInBundle" values(9, 2);
insert into "ProductInBundle" values(9, 4);
insert into "ProductInBundle" values(9, 6);
insert into "ProductInBundle" values(9, 9);

insert into "ProductInBundle" values(10, 2);
insert into "ProductInBundle" values(10, 4);
insert into "ProductInBundle" values(10, 8);
insert into "ProductInBundle" values(10, 9);

insert into "ProductInBundle" values(11, 2);
insert into "ProductInBundle" values(11, 4);
insert into "ProductInBundle" values(11, 6);
insert into "ProductInBundle" values(11, 9);
insert into "ProductInBundle" values(11, 10);

insert into "ProductInBundle" values(12, 2);
insert into "ProductInBundle" values(12, 4);
insert into "ProductInBundle" values(12, 7);
insert into "ProductInBundle" values(12, 8);
insert into "ProductInBundle" values(12, 9);
insert into "ProductInBundle" values(12, 10);

insert into "ProductInBundle" values(13, 2);
insert into "ProductInBundle" values(13, 4);
insert into "ProductInBundle" values(13, 5);
insert into "ProductInBundle" values(13, 8);
insert into "ProductInBundle" values(13, 9);
insert into "ProductInBundle" values(13, 10);

-- vSphere Standard can be either in Started/Standard Bundle
-- By default from UM 3.6 it will be under Standard Bundle, to switch to Starter Bundle a script is used
insert into "ProductInBundle" values(7, 3);

INSERT INTO "VcopsLicenseName"
    (id, name)
SELECT nextval('"s_VcopsLicenseName_id"'), 'Not yet discovered'
WHERE
    NOT EXISTS (
        SELECT * FROM "VcopsLicenseName"
    );

INSERT INTO "VcacLicenseName"
    (id, name)
SELECT nextval('"s_VcacLicenseName_id"'), 'Not yet discovered'
WHERE
    NOT EXISTS (
        SELECT * FROM "VcacLicenseName"
    );

create view "HostLatestHistory" as
    select * from "HostHistory" a where exists(
        select * from (
            select distinct "hostId", max(time) over (partition by "hostId") latest from "HostHistory"
        ) b where a.time = b.latest and a."hostId" = b."hostId"
    );

CREATE VIEW "HostLatest" AS
    SELECT
        h.id,
        hl.id "hhId",
        h.uuid,
        h."vcServerId",
        moref,
        time,
        "nextTime",
        "fullName",
        version,
        name,
        "licenseId",
        "memSize"
    FROM "Host" h
        JOIN "HostLatestHistory" hl ON h.id = hl."hostId";

create view "LicenseHostCount" as
    select l.id, l.name, l.code, count(l.code) as "numHosts" from "License" l, "HostLatestHistory" h
        where h."licenseId" = l.id
        group by l.id, l.name, l.code;

create view "VmLatestHistoryId" as
    select id from "VmHistory" a where exists(
        select * from (
            select distinct "vmId", max(time) over (partition by "vmId") latest from "VmHistory"
        ) b where a.time = b.latest and a."vmId" = b."vmId"
    );

create function betweenco(timestamp, timestamp, timestamp) returns boolean as
    'select $1 >= $2 and $1 < $3'
    language SQL immutable;

create function capped(bigint, bigint) returns bigint as
    'select case when $2 is not null and $1 > $2 then $2 else $1 end'
    language SQL
    immutable;

create function ym(int, int) returns int as
    'select $1 * 100 + $2'
    language SQL
    immutable;

create function year(timestamp) returns int as
    'select extract(year from $1) :: int'
    language SQL
    immutable;

create function month(timestamp) returns int as
    'select extract(month from $1) :: int'
    language SQL
    immutable;

CREATE TYPE "VmHistory4Report" AS
("vmId" integer, "vcId" integer, "vcName" varchar(128), "guestHostName" varchar(128),
 "instanceUuid" varchar(128), "moref" varchar(128), "reportExclusionReason" integer, "vcIP" varchar(128),
 "vmhId" integer, "hostId" integer, "connState" integer, "powerState" integer, "vhTime" timestamp, "vhNextTime" timestamp,
 "vmMemSize" integer,"vmMemRes" integer, "cpuCount" integer, "vrops" boolean, "hhName" varchar(128), "hhMemSize" bigint,
 "licenseId" integer, "hhTime" timestamp, "hhNextTime" timestamp, "nsx" integer, "vcd" boolean, "licenseCategoryId" integer,
 "licenseName" varchar(128), "intervalStart" timestamp, "intervalEnd" timestamp, "billingMem" integer);

CREATE OR REPLACE FUNCTION "vm_history"(report_starttime timestamp, report_endtime timestamp, license_ids int[])
    RETURNS SETOF "VmHistory4Report"
AS $BODY$
SELECT v.id                  "vmId",
       v."vcServerId"        "vcId",
       v."vcName",
       v."guestHostName",
       v."instanceUuid",
       v."moref",
       v."reportExclusionReason",
       vc."host"             "vcIP",
       vh.id                 "vmhId",
       vh."hostId",
       vh."connState",
       vh."powerState",
       vh.time               "vhTime",
       vh."nextTime"         "vhNextTime",
       vh."vmMemSize",
       vh."vmMemRes",
       vh."cpuCount",
       vh."vrops",
       hh."name"             "hhName",
       hh."memSize"          "hhMemSize",
       hh."licenseId",
       hh.time               "hhTime",
       hh."nextTime"         "hhNextTime",
       vh."nsxUsageLicense",
       hh."vcd",
       hh."licenseCategoryId",
       l."name"               "licenseName",
       greatest(vh.time, hh.time)                         "intervalStart",
       least(vh."nextTime", hh."nextTime")                "intervalEnd",
       greatest(vh."vmMemSize"/2, vh."vmMemRes")          "billingMem"
FROM "Vm" v, "VmHistory" vh, "HostHistory" hh, "License" l, "VcServer" vc
WHERE v.id = vh."vmId" AND v."vcServerId" = vc.id AND vh."hostId" = hh."hostId"
AND hh."licenseId" = l.id AND hh."licenseCategoryId" = 1
AND (vh.time, vh."nextTime") overlaps (report_starttime, report_endtime)
AND (hh.time, hh."nextTime") overlaps (report_starttime, report_endtime)
AND (vh.time, vh."nextTime") overlaps (hh.time, hh."nextTime")
AND vh."connState" = 1 AND vh."powerState" = 1
AND ((array_length(license_ids, 1) is null) or (hh."licenseId" = any (license_ids)))
AND v."reportExclusionReason" is null
$BODY$
      LANGUAGE sql;

CREATE TYPE "BillableVmHistory4Report" AS
("vmId" integer, "vcId" integer, "vcName" varchar(128), "guestHostName" varchar(128), "instanceUuid" varchar(128),
 "moref" varchar(128),"reportExclusionReason" integer, "vcIP" varchar(128), "vmhId" integer, "hostId" integer,
 "connState" integer, "powerState" integer, "vhTime" timestamp, "vhNextTime" timestamp, "vmMemSize" integer, "vmMemRes" integer,
 "cpuCount" integer, "vrops" boolean, "hhName" varchar(128), "hhMemSize" bigint, "licenseId" integer,
 "hhTime" timestamp, "hhNextTime" timestamp, "nsx" integer, "vcd" boolean, "licenseCategoryId" integer, "licenseName" varchar(128),
 "intervalStart" timestamp, "intervalEnd" timestamp, "billingMem" integer, "intervalSeconds" double precision,
 "billedVramMbhActual" double precision, "billedVramMbh" double precision);

CREATE FUNCTION "billable_vm_history"(timestamp, timestamp, license_ids int[], int)
    RETURNS SETOF "BillableVmHistory4Report" AS $BODY$
SELECT v.*, v."billingMem" * v."intervalSeconds" / 3600 "billedVramMbhActual",
       capped(v."billingMem", $4) * v."intervalSeconds" / 3600 "billedVramMbh"
FROM (SELECT *,
      extract(epoch from least("intervalEnd", $2, now()) - greatest("intervalStart", $1)) "intervalSeconds"
      FROM vm_history($1, $2, $3) vhl
     ) v
$BODY$
    LANGUAGE sql;

create function customer_summary(timestamp, timestamp, int, license_ids int[])
  returns table(id int, "customerName" varchar, "licenseName" varchar,
    sbm double precision, sbmc double precision)
  as $BODY$
select c.id, c.name "customerName", vhl."licenseName",
    sum(vhl."billedVramMbhActual") sbm,
    sum(vhl."billedVramMbh") sbmc
from billable_vm_history($1, $2, license_ids, $3) vhl
left join "CustomerVm" cv on cv."vmId" = vhl."vmId"
left join "Customer" c on c.id = cv."customerId"
group by c.id, c.name, vhl."licenseName"
$BODY$
    language sql;

create function hosts_license_summary(vcServer int, report_starttime timestamp, report_endtime timestamp)
  returns table("licenseName" varchar, "licenseKey" varchar, "billingCategory" varchar)
  as $BODY$
select distinct l."name", l."code", lc."name"
from "Host" h, "HostHistory" hh, "License" l, "LicenseCategory" lc
where h."vcServerId" = vcServer and h."id" = hh."hostId"
and   (hh."time", hh."nextTime") overlaps (report_starttime, report_endtime)
and   hh."licenseId" = l."id" and hh."licenseCategoryId" = lc."id"
$BODY$
  language sql;

create function vm_counts_by_vc(timestamp, timestamp, license_ids int[], int)
  returns table("vcServerId" int, count bigint)
  as $BODY$
select bvh."vcId", count(distinct bvh."vmId")
from billable_vm_history($1, $2, license_ids, $4) bvh
group by bvh."vcId"
having count(distinct bvh."vmId") > 0
$BODY$
    language sql;

create function vm_counts_by_vcd(timestamp, timestamp, license_ids int[], int)
  returns table("vcdServerId" int, count bigint)
  as $BODY$
select vsc."vcdServerId", count(distinct bvh."vmId")
from billable_vm_history($1, $2, license_ids, $4) bvh
join "CustomerVm" cv on cv."vmId" = bvh."vmId"
join "VcdServerCustomer" vsc on vsc."customerId" = cv."customerId"
group by vsc."vcdServerId"
having count(distinct bvh."vmId") > 0
$BODY$
    language sql;

CREATE TYPE "VsanSpaceUsage" AS
("vcServerId" integer, "clusterId" integer, "vsanLicCategory" integer,
 "usedTotalMb" bigint, "intervalSeconds" double precision);

CREATE OR REPLACE FUNCTION "vsan_space_usage_std_license"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 1
      union
      SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = false AND "erasureCoding" = false) AND
      ("stretchedClusterEnabled" = false AND "iopsLimitUsed" = false)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_adv_license"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 2
      union
      SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = true OR "erasureCoding" = true) AND
      ("stretchedClusterEnabled" = false AND "iopsLimitUsed" = false)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_std_with_addon"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = false AND "erasureCoding" = false) AND
      ("stretchedClusterEnabled" = true OR "iopsLimitUsed" = true)) v
$BODY$
    LANGUAGE sql;

CREATE OR REPLACE FUNCTION "vsan_space_usage_adv_with_addon"(timestamp, timestamp)
    RETURNS SETOF "VsanSpaceUsage" AS $BODY$
SELECT v."vcServerId", v."clusterId", v."vsanLicCategory", v."usedTotalMb",
       extract(epoch from least(v."endTime", $2, now()) - greatest(v."startTime", $1)) "intervalSeconds"
FROM (SELECT c.*
      FROM "ClusterVsanHistory" c
      WHERE ("startTime", "endTime") overlaps ($1, $2)
      AND "vsanLicCategory" = 3 AND
      ("deDupEnabled" = true OR "erasureCoding" = true) AND
      ("stretchedClusterEnabled" = true OR "iopsLimitUsed" = true)) v
$BODY$
    LANGUAGE sql;

CREATE TYPE "VsanClusterInterval" AS
("vcServerId" integer, "clusterId" integer, "vsanLicCategory" integer, "usedTotalMb" bigint,
 "startTime" timestamp, "endTime" timestamp, "deDupEnabled" boolean, "erasureCoding" boolean,
 "stretchedClusterEnabled" boolean, "iopsLimitUsed" boolean
);

CREATE OR REPLACE FUNCTION "vsan_cluster_history"(timestamp, timestamp)
    RETURNS SETOF "VsanClusterInterval" AS $BODY$
SELECT "vcServerId", "clusterId", "vsanLicCategory", "usedTotalMb", "startTime", "endTime",
       "deDupEnabled", "erasureCoding", "stretchedClusterEnabled", "iopsLimitUsed"
FROM   "ClusterVsanHistory"
WHERE  ("startTime", "endTime") overlaps ($1, $2)
$BODY$
    LANGUAGE sql;

create view "AllHostnames" as
select host from "DesktoneServer"
union select host from "LdapServer"
union select host from "NsxServer"
union select host from "VcacServer"
union select host from "VcdServer"
union select host from "VcopsServer"
union select host from "VcServer";

-- New objects for 3.6.0
CREATE TABLE "KeyValueStore"(
   "key" varchar(255) NOT NULL PRIMARY KEY,
   "value" varchar(512) NOT NULL,
   "createDate" timestamp NOT NULL,
   "changedDate" timestamp,
   "note" varchar(255)
);

CREATE TABLE "KeyValueStoreHistory"(
   "id" bigint NOT NULL PRIMARY KEY,
   "key" varchar(255) NOT NULL,
   "oldValue" varchar(512),
   "newValue" varchar(512),
   "createDate" timestamp NOT NULL,
   "note" varchar(255)
);
CREATE TABLE "UnManagedVcVrops" (
    "id" integer primary key not null,
    "vcUuid" varchar(255) not null,
    "vropsServerId" integer not null,
    "timestamp" timestamp not null,
    "vmCount" integer not null
  );
CREATE SEQUENCE "s_UnManagedVcVrops_id";
alter table "UnManagedVcVrops" add constraint "UnManagedVcVropsFK1" foreign key ("vropsServerId") references "VcopsServer"("id");
CREATE TABLE "VcdDR" (
    "id" integer primary key not null,
    "vcdId" integer not null,
    "orgName" varchar(255) not null,
    "orgId" varchar(255) not null,
    "numVms" integer not null,
    "collectTime" timestamp not null
  );
CREATE SEQUENCE "s_VcdDR_id";
alter table "VcdDR" add constraint "VcdDRFK1" foreign key ("vcdId") references "VcdServer"("id");

CREATE SEQUENCE "s_KeyValueStoreHistory_id"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

create table IF NOT EXISTS "VcavServer" (
    "id" integer primary key not null,
    "active" boolean not null,
    "version" varchar(128) not null,
    "host" varchar(128) not null,
    "userName" varchar(128) not null,
    "password" varchar(128) not null,
    "changed" timestamp not null,
    "port" integer not null DEFAULT 443
);

create sequence IF NOT EXISTS "s_VcavServer_id";

create unique index IF NOT EXISTS "vCloudAvailabilityHost" on "VcavServer" ("host");

create table IF NOT EXISTS "VcavVmReplication" (
    "vmId" varchar(128) not null,
    "vmName" varchar(128) not null,
    "sourceSite" varchar(128) not null,
    "destinationSite" varchar(128) not null,
    "direction" varchar(128) not null,
    "isMigration" boolean not null default(false),
    "year" integer not null,
    "month" integer not null
);

create unique index IF NOT EXISTS "VcavVmReplication_vmId_sourceSite_isMigration_month_year" on "VcavVmReplication" ("vmId","sourceSite","isMigration", "year","month");