/* Copyright (c) 2017 VMware, Inc. All rights reserved. -- VMware Confidential */
module platform {

    /**
     * This service class is used to create objects of type H5SdkEventApi.
     */
    export class H5SdkEventService {
        $inject = [
              "vcH5ConstantsService",
              "h5SdkThemeService"
        ];

        constructor(private vcH5ConstantsService: any,
                private h5SdkThemeService: H5SdkThemeService) {
        }

        public createH5SdkEventApi(scope: PluginIframeScope): H5SdkEventApiInitializer {
            return new H5SdkEventApiInitializer(scope, this.vcH5ConstantsService, this.h5SdkThemeService);
        }
    }

    /**
     * The class defines APIs related to handling events.
     */
    export interface H5SdkEventApi {
        /**
         * Registers a global refresh handler which is unique for the current iframe.
         * @param callBack - the function will be called each time
         * the user clicks the global refresh button.
         */
        onGlobalRefresh: (callBack: (() => void)) => void;

        /**
         * Registers a theme changed handler which is unique for the current
         * iframe.
         * @param callback - the function will be called each time the theme the
         * plugin should use changes. The callback signature should be
         * <code>function(theme: PluginTheme): void</code>
         */
        onThemeChanged: (callback: ((theme: PluginTheme) => void)) => void;
    }

    /**
     * Initializes the H5 SDK Event APIs.
     */
    export class H5SdkEventApiInitializer {

        public api: H5SdkEventApi;

        private globalRefreshEvent: (() => void) | null = null;
        private onThemeChangedCallback: ((theme: PluginTheme) => void) | null = null;

        constructor(private scope: PluginIframeScope,
              private vcH5ConstantsService: any,
              private h5SdkThemeService: H5SdkThemeService) {

            this.api = {
                onGlobalRefresh: (onGlobalRefreshHandler: (() => void)) => {
                    this.destroyGlobalRefreshEvent();
                    this.globalRefreshEvent = scope.$on(
                        vcH5ConstantsService.DATA_REFRESH_INVOCATION_EVENT, onGlobalRefreshHandler);
                },
                onThemeChanged: (onThemeChangedCallback: ((theme: PluginTheme) => void)) => {
                    if (typeof onThemeChangedCallback !== 'function') {
                        throw new Error('\'callback\' must be a function!');
                    }

                    this.destroyThemeChangedCallback();
                    this.h5SdkThemeService.subscribeToThemeChanged(onThemeChangedCallback);
                    this.onThemeChangedCallback = onThemeChangedCallback;
                }
            };
        }

        /**
         * Destroys event listeners.
         */
        public onIframeUnload(): void {
            this.destroyGlobalRefreshEvent();
            this.destroyThemeChangedCallback();
        }

        /**
         * Destroys event listeners and performs clean-up.
         */
        public destroy(): void {
            this.destroyGlobalRefreshEvent();
            this.destroyThemeChangedCallback();
        }

        /**
         * Unregister the global refresh event handler the plugin iframe has subscribed to.
         */
        private destroyGlobalRefreshEvent(): void {
            if (this.globalRefreshEvent) {
                this.globalRefreshEvent();
                this.globalRefreshEvent = null;
            }
        }

        /**
         * Unregister the plugin theme changed event handler the plugin iframe
         * has subscribed to.
         */
        private destroyThemeChangedCallback(): void {
            if (this.onThemeChangedCallback) {
                this.h5SdkThemeService.unsubscribeFromThemeChanged(this.onThemeChangedCallback);
                this.onThemeChangedCallback = null;
            }
        }
    }

    angular
        .module("com.vmware.platform.ui")
        .service("h5SdkEventService", H5SdkEventService);
}
