/* Copyright (c) 2017 VMware, Inc. All rights reserved. */

module platform {

   import PluginIframeScope = platform.PluginIframeScope;
   import RemotePluginSingleInstanceExtensionContext = platform.RemotePluginSingleInstanceExtensionContext;
   import ActionEvaluation = com.vmware.vise.actionsfw.ActionEvaluation;

   /**
    * Configuration object describing a modal view.
    */
   export interface ModalConfig {
      url: string;
      title?: string;
      size?: {
         width: number,
         height: number
      };
      // Show/hide of 'x' button of a modal.
      closable?: boolean;

      /**
       * An optional callback that is invoked when the modal is closed.
       * If the modal is closed through the API, plugin-specific data may be provided,
       * which is provided to this callback.
       */
      onClosed?: (result?: any) => void;

      /**
       * An optional field to store an array of context objects that can be passed
       * to the modal when it is opened.
       */
      contextObjects?: any[];

      /**
       * An optional additional data that can be passed
       * to the modal when it is opened.
       */
      customData?: any;
   }

   export interface DynamicModalConfig {
      title?: string;
      height?: number;
   }
   /**
    * The class defines APIs related to the modal dialogs.
    */
   export class H5SdkModalApi {

      constructor(pluginIframeScope: PluginIframeScope,
            $rootScope: any,
            h5SdkModalService: H5SdkModalService) {

         this.open = function (config: ModalConfig): void {
            h5SdkModalService.openModal(
                  config,
                  pluginIframeScope.remotePluginExtensionContext
            );
         };

         this.close = function (result?): void {
            if (pluginIframeScope.context && pluginIframeScope.context.closeModal) {
               pluginIframeScope.context.closeModal(result);
            }
         };

         this.setOptions = function (newModalConfig: DynamicModalConfig): void {
            if (!newModalConfig) {
               return;
            }
            if (pluginIframeScope.context && pluginIframeScope.context.setOptions) {
               pluginIframeScope.context.setOptions(newModalConfig);
            }

            $rootScope.$apply();
         };

         this.getCustomData = function (): any {
            return (pluginIframeScope.context && pluginIframeScope.context.customData) ?
                  pluginIframeScope.context.customData : null;
         };
      }

      open: (config: ModalConfig) => void;

      /**
       * Closes the modal, from which this method is invoked.
       * @param result Arbitrary object that is passed to the onClosed callback of the
       * view that opened the modal.
       */
      close: (result?: any) => void;

      /**
       * Sets the options passed in the ModalConfig parameter
       * @param ModalConfig the new parameters for the modal
       */
      setOptions: (newModalConfig: DynamicModalConfig) => void;

      /**
       * Returns the custom data with which the modal was opened
       * or null if there was no custom data passed.
       * @returns object of custom defined data
       */
      getCustomData: () => any;
   }

   export class H5SdkModalService {
      $inject = ["clarityModalService", "logService", "$rootScope"];

      // The default width of the modal is 24rem (576px) which is Clarity recommended width
      // Width sizes according to Clarity
      private static ONE_REM = 24;
      private static DEFAULT_WIDTH = H5SdkModalService.ONE_REM * 24; // medium Clarity width
      private static MIN_WIDTH = H5SdkModalService.ONE_REM * 12; // small Clarity width
      private static MAX_WIDTH = H5SdkModalService.ONE_REM * 48; // xl Clarity width

      private static DEFAULT_HEIGHT = H5SdkModalService.ONE_REM * 13.5;

      private logger: any;

      constructor(private clarityModalService: any,
            logService: any,
            private $rootScope: any) {
         this.logger = logService('h5SdkModalService');
      }

      openModal(config: ModalConfig,
            remotePluginExtensionContext?: RemotePluginSingleInstanceExtensionContext): void {

         let width: number = H5SdkModalService.DEFAULT_WIDTH;
         if (config.size && config.size.width) {
            if (config.size.width < H5SdkModalService.MIN_WIDTH) {
               width = H5SdkModalService.MIN_WIDTH;
               this.logger.info("Setting modal's width to " + H5SdkModalService.MIN_WIDTH +
                     ", because it cannot be smaller than " + H5SdkModalService.MIN_WIDTH);
            } else if (config.size.width > H5SdkModalService.MAX_WIDTH) {
               width = H5SdkModalService.MAX_WIDTH;
               this.logger.info("Setting modal's width to " + H5SdkModalService.MAX_WIDTH +
                     ", because it cannot be greater than " + H5SdkModalService.MAX_WIDTH);
            } else {
               width = config.size.width;
            }
         }

         let height: number = H5SdkModalService.DEFAULT_HEIGHT;
         if (config.size && config.size.height) {
            height = config.size.height;
         }

         const title: string = config.title ? config.title.trim() : '';
         const closable: boolean =
               (config.closable !== null && config.closable !== undefined) ?
                     !!config.closable : true;

         const htmlModalOptions: any = {
            title: title,
            pluginSize: { width: width, height: height },
            closable: closable,
            url: config.url,
            iframeContext: {
               closeModal: closeModal,
               contextObjects: config.contextObjects,
               setOptions: setOptions,
               customData: config.customData
            },
            remotePluginExtensionContext: remotePluginExtensionContext
         };

         const actionEval: ActionEvaluation = {
            action: {}
         };
         const modalTemplateUrl: string =
               'resources/ui/views/plugins/pluginModalTemplate.html';
         const modalScope: any = this.clarityModalService.openModal(actionEval, null,
               htmlModalOptions, modalTemplateUrl);
         // Callback that is invoked when the user dismisses the modal by clicking the X
         // or the ESC button.
         modalScope.onCancelModal = (result) => {
            if (config.onClosed) {
               try {
                  config.onClosed(result);
               } catch (e) {
                  this.logger.error("Error occurred while invoking the onClosed callback: ", e);
               }
            }
         };

         function closeModal(result): void {
            modalScope.closeModal(result);
         }

         function setOptions(options: DynamicModalConfig): void {
            if (options.title) {
               modalScope.modalOptions.dialogData.title = options.title;
            }
            if (options.height) {
               modalScope.modalOptions.dialogData.pluginSize.height = options.height;
            }
         }
      }

      createH5SdkModalApi(pluginIframeScope: PluginIframeScope): H5SdkModalApi {
         return new H5SdkModalApi(
               pluginIframeScope,
               this.$rootScope,
               this
         );
      }
   }

   angular
         .module("com.vmware.platform.ui")
         .service('h5SdkModalService', H5SdkModalService);
}
